unit TelephonUnit;

{ Diese Unit enthlt ein einfaches TCP-Telephon, das einen bidirektionalen
  Austausch von Strings zwischen zwei Computern realisiert.

  Jedes TCP-Telephon besteht aus einem Client und einem Server. Eine
  Verbindung zwischen zwei Telephonen besteht aus einem Hin-Kanal (vom Client
  des Anrufers zum Server des Angerufenen) und einem Rck-Kanal (vom Client
  des Angerufenen zum Server des Anrufers). Kommt nach dem Einrichten des
  Hin-Kanals kein Rck-Kanal zustande, wird die Verbindung insgesamt wieder
  getrennt.

  Alle vom Betriebssystem generierten Fehlermeldungen werden unterdrckt.
  Statusmeldungen des Programms erscheinen in einem eigenen modalen Fenster.
  Auch der Zustand "Besetzt" ist (mit Hilfe einer Timer-Komponente)
  ordentlich implementiert.

  In den folgenden Kommentaren werden die Bezeichnungen "Hin-Kanal" und
  "Rck-Kanal" stets aus der Sicht des *lokalen* Telephons benutzt.
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Buttons, ExtCtrls, ComCtrls, ScktComp, StdCtrls;

type
  TForm1 = class(TForm)
    StatusBar1: TStatusBar;
    Panel1: TPanel;
    SB_Exit: TSpeedButton;
    ClientSocket1: TClientSocket;
    EditIP: TEdit;
    SB_Connect: TSpeedButton;
    TalkMemo: TMemo;
    ServerSocket1: TServerSocket;
    BtnSend: TButton;
    Btn_Clear: TButton;
    Timer1: TTimer;
    Label1: TLabel;
    Edit1: TEdit;
    procedure SB_ExitClick(Sender: TObject);
    procedure SB_ConnectClick(Sender: TObject);
    procedure ClientSocket1Connect(Sender: TObject; Socket: TCustomWinSocket);
    procedure ClientSocket1Disconnect(Sender: TObject; Socket: TCustomWinSocket);
    procedure ClientSocket1Error(Sender: TObject; Socket: TCustomWinSocket; ErrorEvent: TErrorEvent; var ErrorCode: Integer);
    procedure ServerSocket1ClientConnect(Sender: TObject;
      Socket: TCustomWinSocket);
    procedure ServerSocket1ClientError(Sender: TObject;
      Socket: TCustomWinSocket; ErrorEvent: TErrorEvent;
      var ErrorCode: Integer);
    procedure ServerSocket1ClientDisconnect(Sender: TObject;
      Socket: TCustomWinSocket);
    procedure ServerSocket1ClientRead(Sender: TObject;
      Socket: TCustomWinSocket);
    procedure BtnSendClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Btn_ClearClick(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);    // um "Besetzt" zu erkennen
  private
    { Private declarations }
    procedure TCPErrorProc(ErrorEvent: TErrorEvent; var ErrorCode: Integer);
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation

uses WinSock; { fr die WSAE..-TCP/IP-Fehler-Konstanten in TCPErrorProc }

{$R *.DFM}

{========== Initialisierung ==========================}

procedure TForm1.FormCreate(Sender: TObject);
  begin
  ServerSocket1.Open;
  end;

{========== Verbindung aufbauen ======================}

procedure TForm1.SB_ConnectClick(Sender: TObject);
  begin
  With ClientSocket1 do
    If Active then begin        { Verbindung trennen }
      Close;
      Address := '';
      SB_Connect.Hint := 'Verbinden';
      end
    else begin                  { Verbindung herstellen }
      Address := EditIP.Text;
      Open;
      SB_Connect.Hint := 'Trennen';
      end;
  end;

{========== Client-Socket-Ereignisse ==================}

procedure TForm1.ClientSocket1Connect
                   (Sender: TObject; Socket: TCustomWinSocket);
  { wird aufgerufen, nachdem mein Client den
    Hin-Kanal erfolgreich eingerichtet hat    }
  begin
  With StatusBar1 do begin
    Panels[0].Text := 'Connect';
    Panels[1].Text := Socket.RemoteAddress;
    Panels[2].Text := IntToStr(Socket.RemotePort);
    Panels[3].Text := Socket.LocalAddress;
    Panels[4].Text := IntToStr(Socket.LocalPort);
    end;
  SB_Connect.Down := True;
  Timer1.Enabled  := True;
  {  Dies startet den Timer. Sptestens nach der eingestellten Zeit (2 s)
     wird die Timer1Timer-Prozedur aufgerufen. Dort wird berprft, ob
     inzwischen der komplementre Rckkanal zur TServerSocket-Komponente
     eingerichtet wurde. Falls nicht, wird angenommen, dass der Angerufene
     schon mit jemand anderem spricht. In diesem Fall wird die Verbindung
     der TClientSocket-Komponente mit dem entfernten Server wieder abgebaut
     und eine entsprechende Fehlermeldung ausgegeben. }
  end;

procedure TForm1.ClientSocket1Error
                   (Sender: TObject; Socket: TCustomWinSocket;
                    ErrorEvent: TErrorEvent; var ErrorCode: Integer);
  begin
  TCPErrorProc(ErrorEvent, ErrorCode);
  SB_Connect.Down := False;
  BtnSend.Enabled := False;
  end;

procedure TForm1.ClientSocket1Disconnect
                   (Sender: TObject; Socket: TCustomWinSocket);
  { wird aufgerufen, nachdem mein Client den
    Hin-Kanal erfolgreich abgebaut hat       }
  begin
  With StatusBar1 do begin
    Panels[0].Text := 'Disconnect';
    Panels[1].Text := '';
    Panels[2].Text := '';
    end;
  SB_Connect.Down := False;
  BtnSend.Enabled := False;
  end;

{============= ServerSocket-Ereignisse ==============}

procedure TForm1.ServerSocket1ClientConnect
                    (Sender: TObject; Socket: TCustomWinSocket);
  { wird aufgerufen, nachdem mein Server den
    Rck-Kanal erfolgreich hergestellt hat  }
  begin
  If Not ClientSocket1.Active then begin  { Ich *wurde* angerufen ! }
    ClientSocket1.Address := Socket.RemoteAddress;
    ClientSocket1.Open;
    end
  else                                    { Ich *habe* angerufen ! }
    Timer1Timer(Sender);
    { Dieser Aufruf findet statt, nachdemsich  der lokale Server gerade
      erfolgreich mit dem entfernten Client des Angerufenen verbunden hat.
      Der direkte Aufruf hier krzt die Wartezeit von 2000 ms ab. In der
      Timer1Timer-Prozedur wird berprft, ob die beiden Verbindungen
      wirklich zwischen denselben Computern laufen. Falls nicht, wird die
      Verbindung wieder getrennt. }
  end;

procedure TForm1.ServerSocket1ClientRead
                        (Sender: TObject; Socket: TCustomWinSocket);
  { wird aufgerufen, wenn mein Server eine Nachricht
    vom entfernten Client empfangen hat }
  begin
  If Socket.RemoteAddress = ClientSocket1.Address then   { Filterung ! }
    TalkMemo.Lines.Add('Empfangen : ' + Socket.ReceiveText);
  end;

procedure TForm1.ServerSocket1ClientError
                       (Sender: TObject; Socket: TCustomWinSocket;
                        ErrorEvent: TErrorEvent; var ErrorCode: Integer);
  begin
  TCPErrorProc(ErrorEvent, ErrorCode);
  end;

procedure TForm1.ServerSocket1ClientDisconnect
                        (Sender: TObject; Socket: TCustomWinSocket);
  { wird aufgerufen, nachdem der entfernte Client
    den Rck-Kanal erfolgreich getrennt hat    }
  begin
  If ClientSocket1.Active and
    (Socket.RemoteAddress = ClientSocket1.Address) then begin
    ClientSocket1.Close;          // Trennt den zugehrigen Hin-Kanal !
    ClientSocket1.Address := '';
    end;
  { Die berprfung der Adresse ist notwendig, damit nur die jeweils
    "gemeinte" Verbindung getrennt wird: eventuell kann mein lokaler
    Server mehrere Rck-Kanle aktiv haben!
    Wird ein Rck-Kanal getrennt, zu dem ich keinen Hin-Kanal aufgebaut
    habe, dann gibt es nichts zu trennen und es werden keine weiteren
    Aktionen veranlasst. }
  end;

{============= Nachricht bermitteln ==============}

procedure TForm1.BtnSendClick(Sender: TObject);
  begin
  ClientSocket1.Socket.SendText(Edit1.Text);
  TalkMemo.Lines.Add('Gesendet : ' + Edit1.Text);
  end;

{============= Hilfsprozeduren =====================}

procedure TForm1.Timer1Timer(Sender: TObject);
  { Diese Prozedur berprft, ob die ServerSocket-Komponente einen
    passenden Rck-Kanal zum bestehenden Hin-Kanal einrichten konnte.
    Wird eine "paarige" Verbindung vorgefunden, wird der "Senden"-Knopf
    aktiviert. Andernfalls wird angenommen, dass das angerufene Telephon
    besetzt ist, worauf die Verbindung wieder komplett getrennt wird.   }
  begin
  Timer1.Enabled := False;   // Timer abschalten
  If (ServerSocket1.Socket.ActiveConnections = 1) and        // paarige Verbindung
     (ServerSocket1.Socket.Connections[0].RemoteAddress =
      ClientSocket1.Address) then
    BtnSend.Enabled := True  // Senden erlauben
  else
    if ServerSocket1.Socket.ActiveConnections = 0 then begin // Kein Rck-Kanal!
      ClientSocket1.Close;   // Auch Hin-Kanal wieder trennen !
      ShowMessage('Der gewnschte Anschluss ist besetzt !!!');
      end;
  end;

procedure TForm1.TCPErrorProc(ErrorEvent: TErrorEvent; var ErrorCode: Integer);
  var s : String;
  begin
  Case ErrorCode of
     WSAEHOSTDOWN,
     WSAEHOSTUNREACH : s := 'Kein Anschluss unter dieser Nummer!';
     WSAETIMEDOUT    : s := 'Zeitberschreitung !';
     WSAECONNREFUSED : s := 'Gegenstelle antwortet nicht!';
  else
    Case ErrorEvent of
      eeGeneral   : s := 'General TCP error!';
      eeSend      : s := 'TCP send error!';
      eeReceive   : s := 'TCP receive error!';
      eeConnect   : s := 'TCP connect error!';
      eeDisconnect: s := 'TCP disconnect error!';
      eeAccept    : s := 'TCP accept error!';
    else
      s := 'Unknown TCP error (code = ' + IntToStr(ErrorCode) +')';
    end;
  end;
  ShowMessage(s);
  ErrorCode := 0;   // verhindert zustzliche systeminterne Fehlermeldungen
  end;

procedure TForm1.Btn_ClearClick(Sender: TObject);
  begin
  TalkMemo.Clear;    // lscht alle Eintrge im Gesprchsfenster
  end;

{============= Programm beenden ====================}

procedure TForm1.SB_ExitClick(Sender: TObject);
  begin
  with ClientSocket1 do
    if Active then Close;
  with ServerSocket1 do
    if Active then Close;
  Close;
  end;

end.
