UNIT mTVLInt;

interface

//--------------------  ggf Uses-Liste einfgen !  --------------------

uses classes, sysutils, dialogs;

type
   TVLInt = CLASS(TObject)
     protected
        FDigits : TList;
        FIsNegativ : Boolean;
        function AbsCompareWith(CVLI: TVLInt): Integer;
        function GetDigit(nr: Integer) : Integer;
        procedure SetDigit(nr, val: Integer);
        procedure KillLeadingZeros;
        procedure AddAbs(S2: TVLInt);
        procedure SubtrAbs(Sd: TVLInt);
        procedure MultInt(F2: Integer);
        procedure ShiftLeft(d: Integer);
        procedure LoadDivArray(divisor: TVLInt);
        function  DivDigit(dvsr: TVLInt): Integer;
     public
        constructor Create(s: String);
        constructor CreateCopyOf(Source: TVLInt);
        procedure Free;
        function AsString : String;
        function DigitCount : Integer;
        function IsNegativ : Boolean;
        function IsNull : Boolean;
        function IsEqual(CVLI: TVLInt) : Boolean;
        function IsGreaterThan(CVLI: TVLInt) : Boolean;
        function IsGreaterOrEqualThan(CVLI: TVLInt) : Boolean;
        procedure ChangeSign;
        procedure Assign(Source: TVLint);
        procedure Add(S2: TVLInt);
        procedure Subtr(Sd: TVLInt);
        procedure Mult(F2: TVLInt);
        procedure DivBy(dvsr: TVLInt);
        procedure Modulo(dvsr: TVLInt);
        procedure DivMod(dvsr: TVLInt);
        property Digit[nr: Integer]: Integer read GetDigit write SetDigit;
     end;


implementation


const DigitBase : Integer = 1000000000;  { = 10^9, was etwas weniger
                                                   ist als 2^30  }
var   bufdvsr   : Array [0..29] of TVLInt;
         { Wird benutzt, um Divisionen effizienter zu machen. Es enthlt
           Vielfache des aktuellen Divisors: bufdvsr[n] = divisor*(2^n)
           Das Array lebt ber die gesamte Laufzeit (siehe Code am Ende
           der Unit!). Falls es von einer Division benutzt werden soll,
           werden seine Felder nur dann neu beladen, wenn sie noch nicht
           die "richtigen" Zahlen enthalten. Dadurch profitieren speziell
           solche Algorithmen, bei denen die Modulo-Bildung die einzige
           vorkommende Division ist. }

//+---------------------------------------------------------------------
//|         TVLInt: Methodendefinition
//+---------------------------------------------------------------------

//-------- Create (public) ---------------------------------------------
constructor TVLInt.Create(s: String);
  var i, k : Integer;
  begin
  Inherited Create;
  FDigits := TList.Create;
  For i := Length(s) downTo 1 do    // Alle unerwarteten Zeichen killen
    If Not (s[i] in ['+', '-', '0'..'9']) then
      System.Delete(s, i, 1);
  FIsNegativ := s[1] = '-';         // Vorzeichen erkennen und vermerken
  For i := Length(s) downTo 1 do    // Alle "+" und "-" killen
    If s[i] in ['+', '-'] then
      System.Delete(s, i, 1);
  While (Length(s) > 1) and (s[1] = '0') do
    System.Delete(s, 1, 1);         // Alle fhrenden Nullen killen
  i := Length(s) Div 9;
  If Length(s) Mod 9 > 0 then
    i := i + 1;
  If (i > 0)  and (s <> '0') then begin
    FDigits.Count := i; // Gleich so gro wie ntig dimensionieren
    k := 0;
    While Length(s) >= 9 do begin
      Digit[k] := StrToInt(Copy(s, Length(s)-8, 9));   // Stellen in Liste eintragen
      Delete(s, Length(s)-8, 9);
      k := k + 1;
      end;
    If Length(s) > 0 then
      Digit[k] := StrToInt(s);
    end
  else begin
    FIsNegativ := False;
    Digit[0]   := 0;    // Auch "" wird auch als "0" interpretiert !
    end;
  end;

//-------- CreateCopyOf (public) ---------------------------------------
constructor TVLInt.CreateCopyOf(Source: TVLInt);
  begin
  Inherited Create;
  FDigits := TList.Create;
  Assign(Source);           // Daten holen
  end;

//-------- Free (public) -----------------------------------------------
procedure TVLInt.Free;
  begin
  If Assigned(Self) then begin
    FDigits.Free;
    Inherited Free;
    end;
  end;

//-------- Assign (public) ---------------------------------------------
procedure TVLInt.Assign(Source: TVLInt);
  var i : Integer;
  begin
  FDigits.Clear;                      // Aktuellen Inhalt lschen
  FDigits.Count := Source.DigitCount; // Gleich gengend Platz reservieren!
  For i := 0 to Pred(DigitCount) do
    Digit[i] := Source.Digit[i];      // Alle Ziffern bertragen
  FIsNegativ := Source.IsNegativ;     // Vorzeichen bertragen
  end;

//-------- KillLeadingZeros (protected) --------------------------------
procedure TVLInt.KillLeadingZeros;
  // Eine "listen-nahe" Implementierung fr das Lschen
  // fhrender Nullen sorgt fr einen flotten Ablauf.
  begin
  With FDigits do
    While (Count > 1) and (Last = Nil) do
      Delete(Pred(Count));
  end;

//--------- AddAbs (protected) -----------------------------------------
procedure TVLInt.AddAbs(S2: TVLInt);
  var carry,        // bertrag
      ssum,         // Stellensumme
      gsz,          // grere (der beiden) Stellenzahl(en)
      i : Integer;  // Laufvariable
  begin
  carry := 0;
  If Self.DigitCount > S2.DigitCount then
    gsz := Self.DigitCount
  else
    gsz := S2.DigitCount;
  For i := 0 to gsz - 1 do begin
    ssum := Digit[i] + S2.Digit[i] + carry;
    Digit[i] := ssum Mod DigitBase;
    carry    := ssum Div DigitBase;
    end;
  If carry > 0 then
    Digit[gsz] := carry;
  end;

//--------- SubtrAbs (protected) ---------------------------------------
procedure TVLInt.SubtrAbs(Sd: TVLInt);
  var borrow,       // Geborgter Wert
      sdiff,        // Stellendifferenz
      i : Integer;  // Laufvariable
  begin
  borrow := 0;
  i := 0;
  While (i < Sd.DigitCount) or (borrow > 0) do begin
    sdiff := Digit[i] - (Sd.Digit[i] + borrow);
    If sdiff < 0 then begin
      Digit[i] := sdiff + DigitBase;
      borrow   := 1;
      end
    else begin
      Digit[i] := sdiff;
      borrow   := 0;
      end;
    i := i + 1;
    end;
  KillLeadingZeros;
  end;

//-------- SetDigit (protected) ----------------------------------------
procedure TVLInt.SetDigit(nr, val: Integer);
  begin
  If nr < 0 then
    MessageDlg('Negative Stellennummer!', mtError, [mbOk], 0)
  else begin
    If nr >= DigitCount then
      FDigits.Count := nr + 1;
    FDigits.Items[nr] := Pointer(val);
    end;
  end;

//-------- GetDigit (protected) ----------------------------------------
function TVLInt.GetDigit(nr: Integer) : Integer;
  begin
  If nr < 0 then begin
    MessageDlg('Negative Stellennummer!', mtError, [mbOk], 0);
    Result := 0;
    end
  else
    If nr >= DigitCount then
      Result := 0
    else
      Result := Integer(FDigits.Items[nr]);
  end;

//--------- AbsCompareWith (protected) ---------------------------------
function TVLInt.AbsCompareWith(CVLI: TVLInt): Integer;
  var i : Integer;
  begin
  If DigitCount > CVLI.DigitCount then
    Result := 1
  else
  if DigitCount < CVLI.DigitCount then
    Result := -1
  else begin       // Im Fall "DigitCount = CVLI.DigitCount" machen wir die
    Result := 0;   // vorlufige Annahme, dass die beiden Betrge gleich sind;
    i := Pred(DigitCount);
                   // Dann wird der hchst-wertige Unterschied gesucht.
    While (i >= 0) and (digit[i] = CVLI.digit[i]) do
      i := i - 1;
    If i >= 0 then // Falls eine Stelle mit verschiedenen Ziffern gefunden
                   // wurde, zeigt i jetzt auf die hchstwertige Stelle, in
                   // der sich die beiden Betrge unterscheiden
      If digit[i] > CVLI.digit[i] then
        Result := 1
      else
        Result := -1;
    end;
  end;

//--------- MultInt (protected) ----------------------------------------
procedure TVLInt.MultInt(F2: Integer);
  var p       : Int64;
      u, i, k : Integer;
  begin
  u := 0;
  i := 0;
  k := DigitCount - 1;  // Index der hchsten besetzten Stelle
  While i <= k do begin
    p := digit[i] * Int64(F2) + u;
    digit[i] := p MOD DigitBase;
    u := p DIV DigitBase;
    i := i + 1;
    end;
  If u > 0 then
    Digit[DigitCount] := u;
  end;

//--------- ShiftLeft (protected) --------------------------------------
procedure TVLInt.ShiftLeft(d: Integer);
  var i : Integer;
  begin
  If (d > 0) and (Not IsNull) then begin
    For i := DigitCount - 1 DownTo 0 do
      Digit[i + d] := Digit[i];
    For i := d-1 DownTo 0 do
      Digit[i] := 0;
    end;
  end;

//--------- LoadDivArray (protected) -----------------------------------

procedure TVLInt.LoadDivArray(divisor: TVLInt);
  var i : Integer;
  begin
  If Not bufdvsr[0].IsEqual(divisor) then begin
    bufdvsr[0].Assign(divisor);
    For i := 1 to 29 do begin
      bufdvsr[i].Assign(bufdvsr[i-1]);
      bufdvsr[i].MultInt(2);
      end;
    end;
  end;

//--------- DivDigit (protected) ---------------------------------------
function TVLInt.DivDigit(dvsr: TVLInt): Integer;
  var res, zw,
      i      : Integer;
  begin
  LoadDivArray(dvsr);
  res := 0;
  zw  := 536870912; { 2^29 als grte Zweierpotenz unterhalb von DigitBase }
  For i := 29 DownTo 0 do begin
    If Self.IsGreaterOrEqualThan(bufdvsr[i]) then begin
      res := res + zw;
      Subtr(bufdvsr[i]);
      end;
    zw := zw Div 2;
    end;
  Result := res;
  end;

//-------- AsString (public) -------------------------------------------
function TVLInt.AsString : String;
  var ds : String;
      i  : Integer;
  begin
  Result := '';
  For i := 0 to Pred(DigitCount) do begin
    ds := IntToStr(digit[i]);
    While Length(ds) < 9 do ds := '0' + ds;
    Result := ds + Result;
    end;
  While (Length(Result) > 1) and (Result[1] = '0') do
    Delete(Result, 1, 1);
  If IsNegativ then
    Result := '-' + Result;
  end;

//-------- DigitCount (public) -----------------------------------------
function TVLInt.DigitCount : Integer;
  begin
  Result := FDigits.Count;
  end;

//-------- IsNegativ (public) ------------------------------------------
function TVLInt.IsNegativ : Boolean;
  begin
  Result := FIsNegativ;
  end;

//-------- IsNull (public) ---------------------------------------------
function TVLInt.IsNull : Boolean;
  var i : Integer;
  begin
  Result := True;
  i := Pred(DigitCount);
  While Result and (i >= 0) do
    If digit[i] <> 0 then
      Result := False
    else
      i := i - 1;
  end;

//--------- IsEqual (public) -------------------------------------------
function TVLInt.IsEqual(CVLI: TVLInt): Boolean;
  var i : Integer;
  begin
  If (IsNegativ  = CVLI.IsNegativ) and
     (DigitCount = CVLI.DigitCount) then begin
    Result := True;
    i := Pred(DigitCount);
    While Result and (i >= 0) do
      If digit[i] = CVLI.digit[i] then
        i := i - 1
      else
        Result := False;
    end
  else
    Result := False;
  end;

//--------- IsGreaterThan (public) -------------------------------------
function TVLInt.IsGreaterThan(CVLI: TVLInt): Boolean;
  begin
  If IsNegativ <> CVLI.IsNegativ then  // verschiedene Vorzeichen
    Result := CVLI.IsNegativ
  else                                 // gleiche Vorzeichen
    If IsNegativ then
      Result := AbsCompareWith(CVLI) < 0
    else
      Result := AbsCompareWith(CVLI) > 0;
  end;

//--------- IsGreaterOrEqualThan (public) ------------------------------
function TVLInt.IsGreaterOrEqualThan(CVLI: TVLInt): Boolean;
  begin
  If IsNegativ <> CVLI.IsNegativ then  // verschiedene Vorzeichen
    Result := CVLI.IsNegativ
  else                                 // gleiche Vorzeichen
    If IsNegativ then
      Result := AbsCompareWith(CVLI) <= 0
    else
      Result := AbsCompareWith(CVLI) >= 0;
  end;

//--------- ChangeSign (public) ----------------------------------------
procedure TVLInt.ChangeSign;
  begin
  If Not IsNull then
    FIsNegativ := Not IsNegativ;
  end;

//--------- Add (public) -----------------------------------------------
procedure TVLInt.Add(S2: TVLInt);
  var Pu : TVLInt;
  begin
  If IsNegativ = S2.IsNegativ then  // Gleiche Vorzeichen
    AddAbs(S2)
  else                              // Verschiedene Vorzeichen
    Case Self.AbsCompareWith(S2) of
      1 : SubtrAbs(S2);
      0 : begin
          FDigits.Count := 1;
          FDigits.Items[0] := Nil;
          FIsNegativ := False;
          end;
     -1 : begin
          Pu := TVLInt.CreateCopyOf(S2);
          Pu.SubtrAbs(Self);
          Self.Assign(Pu);
          Pu.Free;
          end;
    end;
  end;

//--------- Subtr (public) ---------------------------------------------
procedure TVLInt.Subtr(Sd: TVLInt);
  begin
  If Not Sd.IsNull then begin
    Sd.ChangeSign;    // Gegenzahl basteln, ...
    Add(Sd);          // ... ersatzweise addieren ...
    Sd.ChangeSign;    // ... und Vorzeichen wieder zurcksetzen!
    end;
  end;

//--------- Mult (public) ----------------------------------------------
procedure TVLInt.Mult(F2: TVLInt);
  var s, p : TVLInt;
      i    : Integer;
  begin
  s := TVLInt.Create('0');
  p := TVLInt.Create('0');
  i := F2.DigitCount - 1;
  While i >= 0 do begin
    p.Assign(Self);
    p.MultInt(F2.Digit[i]);
    s.ShiftLeft(1);
    s.Add(p);
    i := i - 1;
    end;
  Self.Assign(s);
  p.Free;
  s.Free;
  end;

//--------- DivBy (public) ---------------------------------------------
procedure TVLInt.DivBy(dvsr: TVLInt);
  { benutzt den Algorithmus der schriftlichen Division }
  var r, q    : TVLInt;
      d, k, i : Integer;
  begin
  q := TVLInt.Create('0');
  If DigitCount >= dvsr.DigitCount then begin
    i := DigitCount - dvsr.DigitCount;
    r := TVLInt.Create('0');
    For k := DigitCount-1 DownTo i+1 do begin
      r.ShiftLeft(1);
      r.Digit[0] := digit[k];
      end;
    end
  else begin
    r := TVLInt.CreateCopyOf(Self);
    i := - 1;
    end;
  While i >= 0 do begin
    r.ShiftLeft(1);
    r.Digit[0] := digit[i];
    d := r.DivDigit(dvsr);  { hinterlsst in r schon den richtigen Rest ! }
    q.ShiftLeft(1);
    q.Digit[0] := d;
    i := i - 1;
    end;
  Assign(q);
  r.Free;
  q.Free;
  end;

//--------- Modulo (public) --------------------------------------------
procedure TVLInt.Modulo(dvsr: TVLInt);
  { benutzt den Algorithmus der schriftlichen Division }
  var r, q    : TVLInt;
      d, k, i : Integer;
  begin
  q := TVLInt.Create('0');
  If DigitCount >= dvsr.DigitCount then begin
    i := DigitCount - dvsr.DigitCount;
    r := TVLInt.Create('0');
    For k := DigitCount-1 DownTo i+1 do begin
      r.ShiftLeft(1);
      r.Digit[0] := digit[k];
      end;
    end
  else begin
    r := TVLInt.CreateCopyOf(Self);
    i := - 1;
    end;
  While i >= 0 do begin
    r.ShiftLeft(1);
    r.Digit[0] := digit[i];
    d := r.DivDigit(dvsr);  { hinterlsst in r schon den richtigen Rest ! }
    q.ShiftLeft(1);
    q.Digit[0] := d;
    i := i - 1;
    end;
  Assign(r);
  r.Free;
  q.Free;
  end;

//--------- DivMod (public) --------------------------------------------
procedure TVLInt.DivMod(dvsr: TVLInt);
  { benutzt die Idee des "Hilfsalgorithmus" aus divialg.htm in
    einer rekursiven Realisierung statt des Algorithmus der
    "normalen" schriftlichen Division;
    VORSICHT !!!  Gefahr des Stack-berlaufs !!!  }

  var res, eins : TVLInt;

  procedure res2digit(d : TVLInt);
    var d2 : TVLInt;
    begin
    d2 := TVLInt.CreateCopyOf(d);
    d2.MultInt(2);
    If IsGreaterThan(d2) then
      res2digit(d2);
    d2.Free;
    res.MultInt(2);
    If IsGreaterOrEqualThan(d) then begin
      res.Add(eins);
      Self.Subtr(d);
      end;
    end;

  begin
  res  := TVLInt.Create('0');
  eins := TVLInt.Create('1');
  If IsGreaterOrEqualThan(dvsr) then
    res2digit(dvsr);
  dvsr.Assign(Self);
  Self.Assign(res);
  eins.Free;
  res.Free;
  end;


//-------------- Unit-Verwaltung ---------------------------------------

procedure InitDivArray;
  var i : Integer;
  begin
  for i := 0 to 29 do
    bufdvsr[i] := TVLInt.Create('0');
  end;

procedure FreeDivArray;
  var i : Integer;
  begin
  for i := 0 to 29 do
    bufdvsr[i].Free;
  end;


initialization
  InitDivArray;

finalization
  FreeDivArray;

end.
