unit xxcode;

{ XXEncode/XXDecode : 3 Byte <=> 24 Bit <=> 4 Crypt-Zeichen }

interface

uses Classes, SysUtils;

type  TByteBlock  = Array of Byte;

      TXXCodec = class(TObject)
        protected
          ccl : String;                  // "C"rypted "C"haracter "L"ist
          lut : Array [0..255] of Byte;  // Look-Up-Table (Invers zu ccl)
          procedure LoadTransformationTables(iCCL: String);

        public                                         
          function Bytes2String(bytes: TByteBlock): String;
          function String2Bytes(cryptText: String): TByteBlock;
          // Lebenslauf:
          constructor Create(iTT: String);
          procedure Free;

          // String-Konversion:
          function EncodeXXString(source : String): String;
            { verschlsselt einen "normalen" String in einen "XX"-kodierten }
          function DecodeXXString(source : String): String;
            { entschlsselt einen "XX"-kodierten String in einen "normalen" }

          // Konversion beliebiger binrer Daten mit Streams:
          function EncryptStream2XXString(MS: TStream; var s : String) : Boolean;
            { verschlsselt die Bytefolge des bergebenen Streams mit der
              "XX"-Kodierung und gibt die verschlsselten Daten im String s
              zurck; das Ergebnis ist genau dann TRUE, wenn die Konversion
              erfolgreich war. }
          function DecryptXXString2Stream(s: String; MS: TStream) : Boolean;
            { entschlsselt die Daten aus dem bergebenen "XX"-kodierten
              String und schreibt die Bytefolge in den Stream MS; zuvor
              wird der Stream geleert; das Ergebnis der Funktion ist
              genau dann TRUE, wenn die Konversion erfolgreich war. }
        end;


implementation

{========= TXXCoDec ==================================}

constructor TXXCoDec.Create(iTT: String);
  begin
  Inherited Create;
  LoadTransformationTables(iTT);
  end;

procedure TXXCoDec.Free;
  begin
  Inherited Free;
  end;


{--------- Interne Hilfs-Methoden --------------------}

procedure TXXCoDec.LoadTransformationTables(iCCL: String);

  function IsValidCCL(s: String): Boolean;
    var i, k : Integer;
    begin
    If Length(s) = 64 then begin
      Result := True;
      For i := 1 to 64 do    // Alle Zeichen zulssig ?
        If (Ord(s[i]) in [0..31, 127]) then begin
          Result := False;
          Exit;
          end;
      For i := 1 to 63 do    // Kein Zeichen doppelt ?
        For k := i+1 to 64 do
          If s[i] = s[k] then begin
            Result := False;
            Exit;
            end;
      end
    else
      Result := False;
    end;

  var i : Integer;
  begin
  // Transformations-Tabelle einrichten:
  If IsValidCCL(iCCL) then  // Nur gltige Daten nach ccl bernehmen,....
    ccl := iCCL
  else                      // sonst: ccl base64-kompatibel initialisieren !
    ccl := 'ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/';

  // Look-Up-Table einrichten (invers zu ccl):
  For i := 0 to 255 do
    lut[i] := 255;
  For i := 1 to Length(ccl) do
    lut[Ord(ccl[i])] := i - 1;
  end;


function TXXCoDec.Bytes2String(bytes: TByteBlock): String;
  var n, i, pu : Integer;
  begin
  // Daten in Blcken zu 3 Byte kodieren
  n := 0;
  Result := '';
  While n <= High(bytes) - 2 do begin
    pu := 0;
    For i := n + 2 downto n do
      pu := pu SHL 8 + bytes[i];
    For i := 0 to 3 do begin
      Result := Result + ccl[Succ(pu and $0000003F)];
      pu := pu SHR 6;
      end;
    n := n + 3;
    end;

  // Rest-Daten bearbeiten
  If n <= High(bytes) then begin
    pu := 0;
    For i := High(bytes) downto n do
      pu := pu SHL 8 + bytes[i];
    For i := 0 to Succ(High(bytes) - n) do begin
      Result := Result + ccl[Succ(pu and $0000003F)];
      pu := pu SHR 6;
      end;
    end;
  end;


function TXXCoDec.String2Bytes(cryptText: String): TByteBlock;
  var msi,    // Maximal string index
      bc,     // Block Count
      a,      // Additional
      pu,     // Data Buffer
      n, i : Integer;
  begin
  // Alle Steuerzeichen (z.B. Zeilenvorschbe!) und Leerzeichen raus:
  For i := Length(cryptText) DownTo 1 do
    If cryptText[i] <= ' ' then Delete(cryptText, i, 1);

  // Ausgabe-Array dimensionieren:
  msi := Length(cryptText);
  bc := msi Div 4;
  If msi Mod 4 > 0 then       // kann niemals 1 sein !!!
    a := Pred(msi Mod 4)
  else
    a := 0;
  n := bc * 3 + a;
  SetLength(Result, n);

  // CryptText in Blcken zu 4 Zeichen dekodieren:
  n := 0;
  a := 4;
  While a <= bc * 4 do begin
    pu := 0;
    For i := a downto a - 3 do
      pu := pu SHL 6 + lut[Ord(cryptText[i])];
    For i := n to n + 2 do begin
      Result[i] := Byte(pu);
      pu := pu SHR 8;
      end;
    n := n + 3;
    a := a + 4;
    end;

  // Rest des CryptTextes bearbeiten:
  Delete(cryptText, 1, a - 4);
  If Length(cryptText) > 0 then begin
    pu := 0;
    For i := Length(cryptText) downto 1 do
      pu := pu SHL 6 + lut[Ord(cryptText[i])];
    For i := n to n + Length(cryptText) - 2 do begin
      Result[i] := Byte(pu);
      pu := pu SHR 8;
      end;
    end;
  end;


{----------- String-Konversion -----------------------}

function TXXCoDec.EncodeXXString(source : String): String;
  var buf : TByteBlock;
      i   : Integer;
  begin
  SetLength(buf, Length(source));
  For i := 0 to High(buf) do
    buf[i] := Byte(source[Succ(i)]);
  Result := Bytes2String(buf);
  end;


function TXXCoDec.DecodeXXString(source : String): String;
  var buf : TByteBlock;
      i   : Integer;
  begin
  buf := String2Bytes(source);
  SetLength(Result, Succ(High(buf)));
  For i := 1 to Succ(High(buf)) do
    Result[i] := Char(buf[Pred(i)]);
  end;

{---------- Stream-Konversion ------------------------}

function TXXCoDec.EncryptStream2XXString(MS: TStream; var s : String): Boolean;
  var outlen : Integer;
      buf    : TByteBlock;
  begin
  If MS.Size > 0 then
    If MS.Size < (MaxInt Div 4) * 3 - 2 then begin
      outlen := ((MS.Size + 2) Div 3) * 4;
      SetLength(s, outlen);
      SetLength(buf, MS.Size);
      MS.Position := 0;       
      MS.Read(buf[0], MS.Size);
      s := Bytes2String(buf);
      Result := True;
      end
    else
      Result := False
  else begin
    s := '';
    Result := True;
    end;
  end;

function TXXCoDec.DecryptXXString2Stream(s: String; MS: TStream): Boolean;
  var buf : TByteBlock;
  begin
  MS.Position := 0;
  MS.Size     := 0;
  buf := String2Bytes(s);
  MS.Write(buf[0], Succ(High(buf)));
  Result := True;
  end;


end.
