﻿unit mTurtle2;

(*  History:

07.01.98 © Sprandel, Ver 1.7

02.06.98 überarbeitet durch Mechling :
  PrintPic-Problem : automatische Unterscheidung zwischen 16- und
  32-Bit-Windows zur Laufzeit (!) und entsprechende Umschaltung
  des PrintPic-Codes nachgerüstet

04.12.01 überarbeitet durch Mechling :
  Typisierte Konstante "new" ersetzt durch Variable "IsNew", die
  im Initialisierungsteil der Unit gesetzt wird; dann läuft der
  Code auch unter Delphi 6 mit den Standard-Einstellungen des
  Compilers!

01.11.09 überarbeitet durch Mechling :
  mit BDS 2006 versuchsweise in ein "ordentliches" Package verpackt,
  allerdings ohne durchschlagenden Erfolg: das Package erwies sich bei
  der Installation als nicht hinreichend netzwerk-tauglich.
______________________________________________________________________

08.01.10  © Sprandel, Mechling, Ver 2.0 :
  mit BDS 2006 als nicht-visuelle Komponente in eine neue Unit verpackt;
  neuer Konstruktor, der als Parameter eine unterlegte PaintBox erwartet;
  dieser Konstruktor muss zur Laufzeit explizit aufgerufen werden.
*)


interface uses
  SysUtils, Windows, Messages, Classes, Graphics, Printers, ExtCtrls, Forms;

Const MaxTurtles = $FF;
Type  TTurtles = array[0..MaxTurtles] of
        record               { private properties of a Turtle }
          xscreen, yscreen,  { screen coordinates of a Turtle (0/0) upper left }
          alfa: Double;      { angle in degrees; north has alfa = 0° }
          PenColor: TColor;  { PenColor of a Turtle }
          PenWidth: Integer; { PenWidth of a Turtle }
        end;

  TTurtle = class(TObject)
  private
    FTurtle  : TTurtles;
    FWriteRel: boolean;      { true <==> PrTxt outputs right to Turtles direction }
    FNum     : Integer;      { number of a Turtle }
    FBeam    : Boolean;      { enable beaming in color finding process }
    FTH      : Integer;      { Turtleheight }
    FGroundColor: TColor;    { color actually under the Turtle }
    FBGColor : TColor;       { background color setting }
    FVisible : Boolean;      { Turtle state affected by HT/ST }
    FWaitTime: Integer;      { Time to wait between 2 Buffer copy outputs in ms}
    FPenDown : Boolean;      { Turtle state affected by PU/PD }
    FBuffer  : TBitmap;      { Internal buffer bmp }
    FPaintBox: TPaintBox;    { Host component, must be a TPaintBox ! }
    function GetPosX  : Integer;           { gets X-Position of the actual Turtle }
    function GetPosY  : Integer;           { gets Y-Position of the actual Turtle }
    function GetWidth : Integer;           { gets the width of the actual drawing area }
    function GetHeight: Integer;           { gets the height of the actual drawing area }
    function GetCol   : TColor;            { gets the actual drawing color }
    procedure SetCol (newCol: TColor);     { sets the actual drawing color }
    function GetLineWidth : Integer;
    procedure SetLineWidth (newWidth: Integer);
    function GetFont: TFont;               { gets Font }
    procedure SetFont(newFont: TFont);     { sets Font }
    procedure SetTurtleNum(Value: Integer);{ initializes a new Turtle and makes it actual}
    procedure SetBG( Value: TColor);       { sets backgrond color for the actual Turtle}
    procedure SetWaitTime( Value: Integer);{ sets the time to wait between 2 outputs }
    procedure Refresh;                     { starts a grafik output cycle }
    procedure PaintPic(Sender: TObject);   { copies the FBitmap image into the FPaintbox }
    procedure DrawTurtle;                  { draws a Turtle}
    procedure NewPos(px, py: Integer);
  public
    constructor Create(AOwner: TPaintBox);
    destructor Destroy; override;
    procedure HOME;  { = moveTo(0,0) & Falfa=0. Turtlecoordinates are relative
                     { to the startposition (0/0) = Homeposition of the Turtle}
    procedure CS;    { clear screen }
    procedure ST;    { show Turtle }
    procedure HT;    { hide Turtle }
    procedure PD;    { pen down }
    procedure PU;    { pen up }
    procedure FD( Value: Double);         { step forward }
    procedure BK( Value: Double);         { step backward }
    procedure LT( Value: Double);         { left turn }
    procedure RT( Value: Double);         { right turn }
    procedure SetHXY(u, v: Double);       { set heading to location (u,v)}
    procedure MoveXY(u, v: Double);       { move Turtle to location (u,v)}
    function  FoundColor(Var dist: Integer; alfa: Double; Col: TColor): Boolean;
   { searches within the distance <dist> and the Turtle-Angle <alfa> the nearest
     <Col>-colored point; if found, then <dist> = distance of such a point }
    procedure Fill(fillcolor: TColor);   { fill with this color}
    procedure PrTxt( Text: string);      { print text right to Turtle }
    procedure SavePic(FileName: string); { save picture (BitMap) }
    procedure LoadPic(FileName: string); { load picture (BitMap) }
    procedure PrintPic(Wmm: Integer);    { print picture; Wmm = width in mm on paper }
    procedure Resize;                    { resizes the TurtleRect }
  published
    property BeamEnabled: boolean read FBeam write FBeam default FALSE;
    property BGColor: TColor read FBGColor write SetBG;            { backgroundcolor }
    property Color: TColor read getCol write setCol;
    property LineWidth: Integer read getLineWidth write setLineWidth;
    property GroundColor: TColor read FGroundColor;   { gets color under the Turtle }
    property Font: TFont read getFont write setFont;
    property TurtleHeight: Integer read FTH write FTH default 12;
    property Num: integer read FNum write SetTurtleNum default 0;
     { Multi-Turtle property; up to 255 Turtles can be accessed by the number <Num>:
     assigning x>=0 to <Num> activates the Turtle number x and makes this Turtle actual
     assigning x<0 to <Num> deactivates the Turtle number x, default Turtle gets actual
     range of <Num>: -Maxturtles..MaxTurtles ( number of the default Turtle is 0) }
    property PosX: Integer read GetPosX;  { Turtle X- & Y-postiton; coordinates are ~}
    property PosY: Integer read GetPosY;  { ~relative to Homeposition(0/0);read only }
    property Width: Integer read GetWidth;
    property Height: Integer read GetHeight;
    property isVisible: boolean read FVisible;
    property WriteRel: boolean read FWriteRel write FWriteRel default false;
    property WaitTime: Integer read FWaitTime write SetWaitTime default 0;
  end;


(*========================================================================================*)

implementation

Const pidiv180 = pi/180;
Var
    XHome, YHome : Integer;                        { rounded Fx, Fy }
    TurtleRect   : TRect;
    IsActive     : array[0..MaxTurtles] of boolean;{ stores Turtles activity states }
    SaveTurtle   : TTurtles;                       { stores last Turtles settings }


constructor TTurtle.Create(AOwner: TPaintBox);
  begin
  inherited Create;
  FPaintBox := AOwner;
  FBuffer   := TBitmap.Create;
  Resize;
  FPaintbox.OnPaint := PaintPic;
  FBuffer.Canvas.Font.Assign(FPaintBox.Canvas.Font);
  FWaitTime := 0;     // default value
  FTH       := 12;    // default value
  FBGColor     := clWhite;  // Background colour
  FGroundColor := FBGColor; // Colour of the point where the turtle is
  FPendown     := TRUE;
  FVisible     := TRUE;
  IsActive[0]  := TRUE;
  CS;
  end;

destructor  TTurtle.Destroy;
  begin
  FBuffer.Free;
  inherited destroy;
  end;


procedure TTurtle.CS;                   { Clear Screen }
  begin
  Home;
  with FBuffer.Canvas do begin
    brush.color := FBGColor;
    FillRect(TurtleRect);
    end;
  DrawTurtle;
  FPenDown := TRUE;  // reset to default value
  Refresh;
  end;

procedure TTurtle.Home;                 { Home }
  Var actualNum, i : integer;
      penstate     : boolean;
  begin
  actualNum := FNum;
  penstate  := FPendown;
  if FPendown then
    pu;
  for i := 0 to MaxTurtles do
    if IsActive[FNum] then begin
      FNum := i;
      MoveXY(0,0);
      LT(FTurtle[FNum].alfa);
      end;
  FNum     := actualNum;
  FPendown := penstate;
  Refresh;
  end;

procedure TTurtle.FD;                  { Forward }
  Var a      : Double;
      Rx, Ry : Integer;
  begin
  DrawTurtle;
  with FTurtle[FNum] do begin
    SaveTurtle[FNum].xscreen := xscreen;
    SaveTurtle[FNum].yscreen := yscreen;
    a := alfa * pidiv180;
    xscreen := xscreen + Value*sin(a);
    yscreen := yscreen - Value*cos(a);
    Rx := round(xscreen);
    Ry := round(yscreen);
    end; {with}
  NewPos(Rx, Ry);
  DrawTurtle;
  Refresh;
  end;

procedure TTurtle.LT;                 { Left Turn }
  Var alfa : Double;
  begin
  DrawTurtle;  {remove old Turtle}
  alfa := FTurtle[FNum].alfa;
  SaveTurtle[FNum].alfa := alfa;
  alfa := alfa - Value;
  if alfa > +180 then
    alfa := alfa - 360
  else if alfa < -180 then
    alfa := alfa + 360;
  FTurtle[FNum].alfa := alfa;
  DrawTurtle;  {redraw new Turtle}
  Refresh;
  end;

procedure TTurtle.BK;                 { Backward }
  begin
  FD(-Value);
  end;

procedure TTurtle.RT;                 { Right Turn }
  begin
  LT(-Value);
  end;

procedure TTurtle.MoveXY(u, v: Double);   { Move to position (x,y) }
  var Rx, Ry : Integer;
  begin
  DrawTurtle;
  with FTurtle[FNum] do begin
    SaveTurtle[FNum].xscreen := xscreen;
    SaveTurtle[FNum].yscreen := yscreen;
    xscreen := XHome + u;
    yscreen := YHome - v;
    Rx := round(xscreen);
    Ry := round(yscreen);
    end;
  NewPos(Rx, Ry);
  DrawTurtle;
  Refresh;
  end;

procedure TTurtle.SetHXY;           { Set heading to position (x,y) }
  Var   dX,dY: Double;
  begin
  DrawTurtle;
  with FTurtle[FNum] do begin
    SaveTurtle[FNum].alfa := alfa;
    dX := (XHome + u) - xscreen;
    dY := yscreen - (YHome - v);
    if dY > 0 then
      alfa :=ArcTan(dX/dY) / pidiv180
    else if dY < 0 then
      alfa := 180 + ArcTan(dX/dY) / pidiv180
    else if dX > 0 then
      alfa := 90
    else if dX < 0 then
      alfa := -90;
  end; {with}
  DrawTurtle;
  Refresh;
  end;

procedure TTurtle.Fill(fillcolor: TColor);    { Fill with this color }
  var xs, ys : Integer;
  begin
  xs := Round(FTurtle[FNum].xscreen);
  ys := Round(FTurtle[FNum].yscreen);
  if isVisible then
    FBuffer.Canvas.brush.Color := fillcolor XOR BGColor
  else
    FBuffer.Canvas.brush.Color := fillcolor;
  DrawTurtle;
  with FBuffer.Canvas do
    floodfill(xs, ys, pen.Color, fsBorder);
  DrawTurtle;
  Refresh;
  end;

procedure TTurtle.HT;               { Hide Turtle }
  begin
  if FVisible then begin
    DrawTurtle;
    FVisible:=false;
    Refresh;
    end;
  end;

procedure TTurtle.ST;               { Show Turtle }
  begin
  if not FVisible then begin
    FVisible:=true;
    DrawTurtle;
    Refresh;
    end;
  end;

procedure TTurtle.PD;               { Pen Down }
  begin
  FPenDown:=true;
  end;

procedure TTurtle.PU;               { Pen Up }
  begin
  FPenDown:=false;
  end;

function TTurtle.FoundColor(Var dist: integer; alfa: Double; Col: TColor): Boolean;
  Var u, v, dx, dy,
      alfa_now, alfa_save : Double;
      steps, D            : Integer;

  procedure beam;
    Var i, OldTH       : integer;
        VState, PState : boolean;
    begin
    VState := FVisible;
    PState := FPendown;
    ST;
    PU;
    OldTH := FTH;
    FTH := FTH div 3;
    DrawTurtle;
    RT(alfa);
    for i := 1 to steps*8 do
      FD(1/8);
    BK(steps);
    LT(alfa);
    DrawTurtle;
    FTH := OldTH;
    FVisible := VState;
    FPendown := PState;
    end;

  begin
  D := abs(dist);
  alfa_save := FTurtle[FNum].alfa;
  alfa_now  := (alfa_save + alfa) * pidiv180;
  dx := sin(alfa_now);
  dy := cos(alfa_now);
  u  := FTurtle[FNum].xscreen;
  v  := FTurtle[FNum].yscreen;
  for steps := 0 to D do begin
    u := u + dx;
    v := v - dy;
    if FBuffer.Canvas.Pixels[round(u),round(v)]=Col then
      break;
    end;
  dist := steps;
  result := steps < D;
  if FBeam then
    beam;
  FTurtle[FNum].alfa := alfa_save;
  end;

procedure TTurtle.PrTxt;
  Var a, b : integer;
      t    : Double;
      N    : string;
      p    : array[0..32] of Char;
      Italics, Underline, Strike : byte;
      hRotatableFont             : HFont;

  procedure PrtTxt(C: TCanvas; x,y: integer);
    Var hOldFont : HFont;
    begin
    x := round(x + 10 * sin(t));
    x := Round(FTurtle[FNum].xscreen) - x;
    y := round(y - 10 * cos(t));
    y := Round(FTurtle[FNum].yscreen) - y;
    with C do begin
      hOldFont    := SelectObject(Handle, hRotatableFont);
      Font.Color  := FBuffer.Canvas.Font.Color;
      Brush.Color := BGColor;
      TextOut(x,y, Text);
      SelectObject(Handle, hOldFont);
      end;
    end;

  begin
  if NOT FWriteRel then
    a := 0
  else
    a := round(FTurtle[Fnum].alfa);
  with FBuffer.Canvas.Font do begin
    N := Name;
    StrPCopy(p,N);
    if fsBold in Style then b := FW_BOLD else b := FW_NORMAL;
    if fsItalic in Style then Italics := 1 else Italics := 0;
    if fsUnderline in Style then Underline := 1 else Underline := 0;
    if fsStrikeOut in Style then Strike := 1 else Strike := 0;
    hRotatableFont:= CreateFont(height, 0, -a*10, 0, b,
                                Italics, Underline, Strike, DEFAULT_CHARSET,
                                0, 0, 0, 0, p);
    end;
  if N = 'System' then
    t := -0.8
  else
    t := (a-60) * pidiv180;
  PrtTxt(FBuffer.Canvas, 0, 0);
  DeleteObject(hRotatableFont);
  Refresh;
  end;

procedure TTurtle.SavePic;
  begin
  DrawTurtle;   { remove }
  FBuffer.SaveToFile(FileName);
  DrawTurtle;   { redraw }
  end;

procedure TTurtle.LoadPic(FileName: string);
  Var BMP: TBitMap;
      nx, ny : Integer;
  begin
  CS;
  HT;
  BMP := TBitMap.Create;
  BMP.LoadFromFile(FileName);
  nx := Width - BMP.Width;
  ny := Height - BMP.Height;
  FBuffer.Canvas.Draw(nx Div 2, ny Div 2, BMP);
  BMP.free;
  FBGColor := FPaintBox.Canvas.Pixels[2,2];
  ST;
  Refresh;
  end;

procedure TTurtle.PrintPic(Wmm: Integer);
  { Code for 32 bit Windows; old Win3 code removed }
  Var w, h   : Integer;
      ReDest : TRect;
  begin
  with Printer do begin
    BeginDoc;
    SetMapMode(Handle, MM_TEXT);
    w := muldiv(Wmm, pagewidth, GetDeviceCaps(Handle, HORZSIZE));
    h := muldiv(w, Height , Width);
    ReDest := bounds((pagewidth-w) div 2, (pageheight-h) div 3, w, h);
    Printer.Canvas.CopyRect(ReDest, FBuffer.Canvas, TurtleRect);
    EndDoc;
    end;
  end;

procedure TTurtle.Resize;
  { Fits the buffer image to the dimensions of the hosting PaintBox;
    the drawing's content will be lost because the buffer is resized;
    a CS command is executed to correctly re-initialize the Buffer.   }
  var nW, nH   : Integer;
  begin
  nW := FPaintBox.Width;
  nH := FPaintBox.Height;
  TurtleRect := Bounds(0, 0, nW, nH);
  FBuffer.Width  := nW;
  FBuffer.Height := nH;
  XHome := nW div 2;
  YHome := nH div 2;
  CS;
  end;


{======= Interne Hilfsprozeduren =================}

procedure TTurtle.DrawTurtle;
  var rsa, rca, a: Double;
  begin
  if FVisible and IsActive[FNum] then
    with FTurtle[FNum], FBuffer.Canvas do begin
      a   := FTurtle[FNum].alfa * pidiv180;
      rsa := FTH * sin(a);
      rca := FTH * cos(a);
      Pen.Width := 1;
      Pen.Color := FBGColor xor clBlack;
      Pen.mode  := pmXor;
      Polyline([Point(round(xscreen+rsa),round(yscreen-rca)),
                Point(round(xscreen+rca),round(yscreen+rsa)),
                Point(round(xscreen-rca),round(yscreen-rsa)),
                Point(round(xscreen+rsa),round(yscreen-rca))]);
      Pen.mode  := pmCopy;
      Pen.Color := PenColor;
      Pen.Width := PenWidth;
      end; {of with}
  end;

procedure TTurtle.Refresh;

  procedure wait(delay : Integer);
    var start: Integer;
    begin
    start := GetTickCount;
    While Integer(GetTickCount) - start < delay do
      Application.ProcessMessages;
    end;

  begin
  if FWaitTime = 0 then
    FPaintBox.Invalidate
  else begin
    PaintPic(Nil);
    Wait(FWaitTime);
    end;
  end;

procedure TTurtle.PaintPic(Sender: TObject);
  begin
  FPaintBox.Canvas.CopyRect(TurtleRect, FBuffer.Canvas, TurtleRect);
  end;

function TTurtle.GetPosX: Integer;
  begin
  result:=round(FTurtle[Num].xscreen - XHome);
  end;

function TTurtle.GetPosY: Integer;
  begin
  result:=round(YHome - FTurtle[FNum].yscreen)
  end;

function TTurtle.GetWidth;
  begin
  Result := FPaintBox.Width;
  end;

function TTurtle.GetHeight;
  begin
  Result := FPaintBox.Height;
  end;

procedure TTurtle.SetBG( Value: TColor);
  begin
  FBGColor:=Value;
  CS;
  end;

function TTurtle.GetCol: TColor;               { gets the actual drawing color }
  begin
  Result := FTurtle[FNum].PenColor;
  end;

procedure TTurtle.SetCol (newCol: TColor);     { sets the actual drawing color }
  begin
  FTurtle[FNum].PenColor := newCol;
  FBuffer.Canvas.Pen.Color := newCol;    // maybe this is not necessary ?
  end;

function TTurtle.GetLineWidth: Integer;
  begin
  Result := FTurtle[FNum].PenWidth;
  end;

procedure TTurtle.SetLineWidth ( newWidth : Integer);
  begin
  FTurtle[FNum].PenWidth := newWidth;
  FBuffer.Canvas.Pen.Width := newWidth;   // maybe this is not necessary ?
  end;

function TTurtle.GetFont: TFont;
  begin
  Result := FBuffer.Canvas.Font;
  end;

procedure TTurtle.SetFont(newFont: TFont);
  begin
  FBuffer.Canvas.Font.assign(newFont);
  end;

procedure TTurtle.SetWaitTime(Value: Integer);
  begin
  Value := Abs(Value);
  if Value <= 2000 then
    FWaitTime := Value
  else
    FWaitTime := 2000;
  end;

procedure TTurtle.SetTurtleNum;
  begin
  if abs(value)<=MaxTurtles then
    if Value >= 0 then
      with FTurtle[Value] do begin
        FNum:=Value;
        if not IsActive[Value] then begin
          IsActive[Value]:=True;
          xscreen:=XHome;
          yscreen:=YHome;
          end; { of inner if }
        end { of with }
    else begin
      if IsActive[-value] then begin
        FNum:=-Value;
        HT;
        end;
      IsActive[-value]:=FALSE;
      FNum:=0;
      end;
  end;

procedure TTurtle.NewPos(px, py: Integer);
  begin
  with FBuffer.Canvas do begin
    pen.color:=FTurtle[FNum].pencolor;
    pen.width:=FTurtle[FNum].penwidth;
    MoveTo(round(SaveTurtle[FNum].xscreen),
           round(SaveTurtle[FNum].yscreen));
    if FPenDown then
      LineTo(px, py)
    else
      MoveTo(px, py);
    FGroundColor:=pixels[px, py];
    end; {of with}
  end;

end.
